/************************************************************************
 * @file: gst_dlt_adapter.c
 *
 * @version: 0.1
 *
 * @description: This source file contains implementation of all the GDA APIs,
 * required for routing the GStreamer logs to DLT Viewer
 *
 * @authors: Athreya Manasa Balaraj, ManasaBalaraj.Athreya@in.bosch.com 2017
 *           Devi Laxmi, Laxmi.Devi@in.bosch.com 2017
 *
 * @copyright (c) 2015 Advanced Driver Information Technology.
 * Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
 * All rights reserved.
 *
 ***********************************************************************/

#include <dlt/dlt.h>
#include <gst/gst.h>
#include <string.h>
#include <stdio.h>
#include <ctype.h>
#include <stdlib.h>
#include <stdbool.h>

#include "gst_dlt_internals.h"

/* To maintain the list of GDA Context, gp_gda_ctx variable is created as
   head reference */
gda_ctx_ll_t *gp_gda_ctx = NULL;

/* When the thread_id is not associated with the any DLT Context,
   logs are traced in GDA context in DLT viewer */
DltContext gda_dlt_ctx;

/* Global GRecMutex for GDA */
static GRecMutex g_gda_mtx;

static bool
gda_check_numbers (const char * str)
{
    if (str == NULL)
    {
        return false;
    }
    while (*str)
    {
        if (isdigit (*str++) == 0)
        {
            return false;
        }
    }
    return true;
}

static bool
gda_match (char * first, const char * second)
{
    if ((first == NULL) || (second == NULL))
    {
        return false;
    }
    if ((first[0] == '\0') && (second[0] == '\0'))
    {
        return true;
    }
    if ((first[0] == '*') && (first[1] != '\0') && (second[0] == '\0'))
    {
        return false;
    }
    if (first[0] == second[0])
    {
        return gda_match ((first + 1), (second + 1));
    }
    if (first[0] == '*')
    {
        return (gda_match ((first + 1), second)) || (gda_match (first, (second + 1)));
    }
    return false;
}

static inline void
gda_unlock_entry(void)
{
    g_rec_mutex_unlock(&g_gda_mtx);
}

static GDA_ER
gda_get_locked_entry_from_id(gda_ctx_ll_t **gda_ctx_inst,const char * dlt_ctx_id)
{
    GDA_ER err = GDA_OK;
    g_rec_mutex_lock(&g_gda_mtx);
    if (gp_gda_ctx == NULL)
    {
        g_print ("GDA :GDA Context is empty\n");
        return GDA_ER_NO_CTX_AVAILABLE;
    }
    *gda_ctx_inst = gp_gda_ctx;
    while (gda_ctx_inst != NULL)
    {
        if(strncmp((*gda_ctx_inst)->ctx->dlt_ctx.contextID, dlt_ctx_id, DLT_ID_SIZE) == 0)
         {
            /* We got the right entry */
            break;
         }
        else if ((*gda_ctx_inst)->next_ctx == NULL)
        {
            DLT_LOG (gda_dlt_ctx, DLT_LOG_ERROR,
                    DLT_STRING("GDA : ContextID did not match any registered dlt contextIDs"));
            err = GDA_ER_CTX_NOT_REGISTERED;
        }
        else
        {
            *gda_ctx_inst = (*gda_ctx_inst)->next_ctx;
        }
    }
    return err;
}

static void
gda_put_entry( gda_ctx_ll_t * p_entry)
{
    gda_ctx_ll_t *gda_ctx_inst = NULL;
    if(gp_gda_ctx == NULL)
    {
        gp_gda_ctx = p_entry;
    }
    else
    {
        gda_ctx_inst = gp_gda_ctx;
        while (gda_ctx_inst->next_ctx != NULL)
        {
            gda_ctx_inst = gda_ctx_inst->next_ctx;
        }
        gda_ctx_inst->next_ctx = p_entry;
    }
}

static int
gda_filter_category_callback(uint32_t service_id, void * inj_data, uint32_t length, void * data)
{
#define PAIR_FILTER ";"
#define CAT_FILTER ":"
    gda_log_list_t  *p_new          = NULL;
    gda_log_list_t  *p_cur          = NULL;
    gda_ctx_ll_t    *gda_inst       = NULL;
    GDA_ER          err             = GDA_OK;
    char            *s_out_filter   = NULL;
    char            *gst_filter     = (char*) inj_data;
    char            *dlt_contextID  = (char*) data;
    uint32_t        category_length = 0;
    length = length;

#ifdef DEBUG
    g_print ("GDA: Injection Message with service_id = 0X%x length = %u\n", service_id, length);
#endif
    /* s_out_filter and s_in_filter are used internally by strtok_r() in order to maintain
       context between successive calls that parse the same string. */
    char *p_out_filter = strtok_r (gst_filter, PAIR_FILTER, &s_out_filter);
    while (p_out_filter != NULL)
    {
        char *s_in_filter = NULL;
        char *p_in_filter = strtok_r (p_out_filter, CAT_FILTER, &s_in_filter);
        if (gda_check_numbers (p_in_filter) == true)
        {
            /* Do Nothing */
            DLT_LOG (gda_dlt_ctx, DLT_LOG_WARN, DLT_STRING("GDA : Wrong usage of Injection"));
        }
        else
        {
            /* New node creation and memory allocation */
            p_new = calloc(1, sizeof(gda_log_list_t));
            if (p_new == NULL)
            {
                DLT_LOG (gda_dlt_ctx, DLT_LOG_WARN, DLT_STRING("GDA : Injection Message Memory allocation fails"));
                return -1;
            }
            while (p_in_filter != NULL)
            {
                if (gda_check_numbers (p_in_filter) == true)
                {
                    p_new->level = (unsigned)(atoi (p_in_filter));
                }
                else
                {
                    category_length = strlen (p_in_filter);
                    if (p_new->category == NULL)
                    {
                        p_new->category = calloc (1, (sizeof(char) * category_length));
                        if (p_new->category == NULL)
                        {
                            free(p_new);
                            p_new = NULL;
                            DLT_LOG (gda_dlt_ctx, DLT_LOG_WARN, DLT_STRING("GDA : Injection Message Memory allocation fails"));
                            return -1;
                        }
                    }
                    strncpy(p_new->category, p_in_filter, category_length);
                }
                p_in_filter = strtok_r (NULL, CAT_FILTER ,&s_in_filter);
            }

            err = gda_get_locked_entry_from_id(&gda_inst, dlt_contextID);
            if(err == GDA_OK && gda_inst != NULL && (service_id > 0XFFF))
            {
                    p_new->next = NULL;
                    if (gda_inst->gda_log_list == NULL)
                    {
                        gda_inst->gda_log_list = p_new;
                    }
                    else
                    {
                        p_cur = gda_inst->gda_log_list;
                        while (p_cur->next != NULL)
                        {
                            p_cur = p_cur->next;
                        }
                        p_cur->next = p_new;
                    }
            }
            else
            {
                if(p_new != NULL)
                {
                    if (p_new->category != NULL)
                    {
                        free(p_new->category);
                        p_new->category = NULL;
                    }
                    free(p_new);
                    p_new = NULL;
                }
            }
            gda_unlock_entry();
        }
        p_out_filter = strtok_r (NULL, PAIR_FILTER, &s_out_filter);
    }
    return 0;
}

static int
gda_compare_thread_id (thread_info_t *thread_info, GThread *thread_id)
{
    uint32_t i;

    for(i = 0; i < thread_info->gda_thread_count; i++)
    {
        if(thread_info->gda_thread_id_arr[i] == thread_id)
        {
            return 1;
        }
    }
  return 0;
}

static gchar *
gda_frame_debug_log_msg (GstDebugCategory * category, const gchar * file,
        const gchar * function, gint line, GObject * object, GstDebugMessage * message)
{
    gchar *obj = NULL;
    gchar *msg = NULL;

    if (object == NULL)
    {
        obj = g_strdup ("(NULL)");
    }
    else
    {
        /*PRQA: Lint Message 774,160 : Boolean within if always evaluates to False.
        The sequence ( { is non standard and is taken to introduce a GNU statement expression*/
        /*lint -save -e774 -e160 As per,Gstreamer open source usage*/
        if ((GST_IS_PAD (object)) && (GST_OBJECT_NAME (object)))
        {
            obj = g_strdup_printf ("<%s:%s>", GST_DEBUG_PAD_NAME (object));
        }
        else if ((GST_IS_OBJECT (object)) && (GST_OBJECT_NAME (object)))
        {
            obj = g_strdup_printf ("<%s>", GST_OBJECT_NAME (object));
        }
        /*lint -restore*/
        else
        {
            obj = g_strdup ("");
        }
    }
    msg = g_strdup_printf ("%p %s %s:%d:%s:%s %s\n",g_thread_self(),
              gst_debug_category_get_name (category), file, line,
              function, obj, gst_debug_message_get (message));

    g_free (obj);
    return msg;
}

static void
gda_get_debug_log (GstDebugCategory * category, GstDebugLevel level,
    const gchar * file, const gchar * function, gint line,
    GObject * object, GstDebugMessage * message, gpointer p_gst_ctx)
{
    gda_ctx_ll_t   *p_inst = NULL;
    gda_log_list_t *p_log  = NULL;
    gchar *msg             = NULL;
    DltContext ctx         = gda_dlt_ctx;
    gboolean msg_handled   = FALSE;
    gboolean context_set   = FALSE;
    gboolean is_single_pipeline = FALSE;
    p_gst_ctx = p_gst_ctx;

    g_rec_mutex_lock(&g_gda_mtx);
    p_inst =  gp_gda_ctx;

    if( p_inst != NULL && p_inst->next_ctx == NULL)
    {
        is_single_pipeline = TRUE;
    }
    while (p_inst != NULL && FALSE == msg_handled)
    {
        if (is_single_pipeline == TRUE)
        {
            ctx = p_inst->ctx->dlt_ctx;
            context_set = true;
        }
        else
        {
            if(p_inst->thread_info != NULL)
            {
                if (gda_compare_thread_id (p_inst->thread_info, g_thread_self()))
                {
                    /* Multiple pipeline case when the thread id matches, else log will be printed in GDAC context */
                    ctx = p_inst->ctx->dlt_ctx;
                    context_set = true;
                }
            }
        }
        if(context_set == true || p_inst->next_ctx == NULL)
        {
            if (dlt_user_is_logLevel_enabled (&ctx, gst2dlt[level]))
            {
                p_log = p_inst->gda_log_list;
                if (p_log != NULL)
                {
                    /* Search the required category by traversing the list */
                    do
                    {
                        if (gda_match (p_log->category, gst_debug_category_get_name (category)) == true)
                        {
                            /* Lint requests typecast */
                            if (((uint32_t)level <= p_log->level) || (p_log->level == 0))
                            {
                                msg = gda_frame_debug_log_msg(category,file, function, line,
                                        object, message);
                                DLT_LOG (ctx, gst2dlt[level], DLT_STRING(msg));
                            }
                            msg_handled = TRUE;
                        }
                        else if(p_log->next == NULL)
                        {
                            msg_handled = TRUE;
                        }
                        else
                        {
                            p_log = p_log->next;
                        }
                    } while (p_log != NULL && FALSE == msg_handled);
                }
                else
                {
                    msg = gda_frame_debug_log_msg(category,file, function, line,
                            object, message);
                    DLT_LOG (ctx, gst2dlt[level], DLT_STRING(msg));
                    msg_handled = TRUE;
                }
            }
            else
            {
                msg_handled = TRUE;
            }
        }
        p_inst = p_inst->next_ctx;
    }
    g_rec_mutex_unlock(&g_gda_mtx);
    g_free (msg);
}

static void
gda_user_set_log_level_callback (char context_id[DLT_ID_SIZE], uint8_t log_level, uint8_t trace_status)
{
    gda_ctx_ll_t *gda_inst = NULL;
    GDA_ER        err      = GDA_OK;
    trace_status = trace_status;

    err = gda_get_locked_entry_from_id(&gda_inst, context_id);
    if(err == GDA_OK && gda_inst != NULL)
    {
        gda_inst->ctx->dlt_log_level = (DltLogLevelType)log_level;
    }
    else
    {
        DLT_LOG (gda_dlt_ctx, DLT_LOG_ERROR,
                            DLT_STRING("GDA : Couldn't set the GStreamer log level"));
    }
    gda_unlock_entry();
    /* Do not call gst_debug_set_threshold_for_name holding a Mutex, Results in deadlock sometimes
     * As gst_debug_set_threshold_for_name can emit a GST Debug message */
    gst_debug_set_threshold_for_name ("*", dlt2gst[log_level]);
}

GDA_ER
gda_register_logging (GDA_CTX * p_gda_ctx, const char * dlt_ctx_id,
                      const char * dlt_ctx_desc, uint32_t injection_service_id)
{
    gda_ctx_ll_t    *p_new    = NULL;
    const char      *p_env    = NULL;
    static uint32_t log_level = GST_LEVEL_ERROR;

    if ((dlt_ctx_id == 0) || (dlt_ctx_id[0] == '\0') || (strlen(dlt_ctx_id) > 4))
    {
        g_print ("GDA: DLT Context ID not set\n");
        return GDA_ER_DLT_CTXID_NOT_SET;
    }
    if (p_gda_ctx == NULL)
    {
        g_print ("GDA: Memory not allocated for  GDA Context\n");
        return GDA_ER_CTX_MEM_NOT_ALLOCATED;
    }
    else
    {
        memset(p_gda_ctx, '\0', sizeof(GDA_CTX));
    }

    p_new = calloc(1, sizeof(gda_ctx_ll_t));
    if (p_new == NULL)
    {
        DLT_UNREGISTER_CONTEXT (p_gda_ctx->dlt_ctx);
        g_print ("GDA : Failed to create GDA Context\n");
        return GDA_ER_NO_MEM;
    }
    p_new->ctx = p_gda_ctx;
    p_new->gda_log_list = NULL;
    p_new->thread_info = NULL;

    g_rec_mutex_lock(&g_gda_mtx);
    if (gp_gda_ctx == NULL)
    {
        gda_put_entry(p_new);
        p_env = g_getenv ("GST_DEBUG");
        if(NULL != p_env)
        {
            uint32_t tmp = 0;
            while (*p_env)
            {
                if (isdigit (*p_env))
                {
                    tmp = (unsigned)(atoi (p_env));
                    log_level = (tmp > log_level) ? tmp : log_level;
                }
                p_env++;
            }
        }
        DLT_REGISTER_CONTEXT_LL_TS (gda_dlt_ctx, GDA_DEFAULT_CTXID, GDA_DEFAULT_CTXID_DES, gst2dlt[log_level], DLT_TRACE_STATUS_DEFAULT);
#ifdef GST_VERSION_1
        gst_debug_add_log_function (gda_get_debug_log, NULL, NULL);
#else
        gst_debug_add_log_function (gda_get_debug_log, NULL);
#endif
        gst_debug_remove_log_function (gst_debug_log_default);

    }
    else
    {
        gda_put_entry(p_new);
    }
    g_rec_mutex_unlock(&g_gda_mtx);
    p_gda_ctx->dlt_log_level  = gst2dlt[log_level];
    p_gda_ctx->dlt_service_id = injection_service_id;

    DLT_REGISTER_CONTEXT_LL_TS (p_gda_ctx->dlt_ctx, dlt_ctx_id, dlt_ctx_desc, gst2dlt[log_level], DLT_TRACE_STATUS_DEFAULT);
    DLT_REGISTER_LOG_LEVEL_CHANGED_CALLBACK(p_gda_ctx->dlt_ctx, gda_user_set_log_level_callback);
    DLT_REGISTER_INJECTION_CALLBACK_WITH_ID (p_gda_ctx->dlt_ctx, p_gda_ctx->dlt_service_id, gda_filter_category_callback, (void*)p_gda_ctx->dlt_ctx.contextID);

    return GDA_OK;
}

GDA_ER
gda_register_thread_to_context (GThread * thread_id, const char * dlt_ctx_id)
{
    gda_ctx_ll_t *gda_ctx_inst = NULL;
    GDA_ER err = GDA_OK;

    if(thread_id == NULL)
    {
        return GDA_ER_THREAD_ID_IS_NULL;
    }
    err = gda_get_locked_entry_from_id(&gda_ctx_inst, dlt_ctx_id);
    if(err == GDA_OK && gda_ctx_inst != NULL)
    {
        if(gda_ctx_inst->thread_info == NULL)
        {
            gda_ctx_inst->thread_info = calloc(1, sizeof(thread_info_t));
            if(gda_ctx_inst->thread_info == NULL)
            {
                gda_unlock_entry();
                return GDA_ER_NO_MEM;
            }
        }
        if(gda_ctx_inst->thread_info->gda_thread_count < MAX_THREADS)
        {
#ifdef DEBUG
            g_print("GDA :Thread %p is registering to context %s\n",  thread_id, dlt_ctx_id);
#endif
            gda_ctx_inst->thread_info->gda_thread_id_arr[(gda_ctx_inst->thread_info->gda_thread_count)] = thread_id;
            gda_ctx_inst->thread_info->gda_thread_count++;
        }
        else
        {
            DLT_LOG (gda_dlt_ctx, DLT_LOG_ERROR, DLT_STRING("GDA : GST Threads max limit per context reached"));
            err = GDA_ER_CTX_THREADS_LIMIT_REACHED;
        }
    }
    gda_unlock_entry();
    return err;
}

GDA_ER
gda_unregister_logging (GDA_CTX * p_gda_ctx)
{
    gda_log_list_t  *p_inst       = NULL;
    gda_log_list_t  *p_cur        = NULL;
    gda_ctx_ll_t    *gda_prev_ctx = NULL;
    gda_ctx_ll_t    *gda_cur_ctx  = NULL;

    g_rec_mutex_lock(&g_gda_mtx);
    if (gp_gda_ctx == NULL)
    {
        g_rec_mutex_unlock(&g_gda_mtx);
        g_print ("GDA :GDA Context is empty\n");
        return GDA_ER_NO_CTX_AVAILABLE;
    }

    gda_prev_ctx = gp_gda_ctx;
    gda_cur_ctx = gp_gda_ctx;

    while (gda_prev_ctx != NULL)
    {
        if (gda_prev_ctx->ctx == p_gda_ctx)
        {
            p_inst = gda_prev_ctx->gda_log_list;
            while (p_inst != NULL)
            {
                if (p_inst->category != NULL)
                {
                    free (p_inst->category);
                    p_inst->category = NULL;
                }
                p_cur = p_inst->next;
                free (p_inst);
                p_inst = p_cur;
            }
            if(gda_prev_ctx->thread_info != NULL)
            {
                free (gda_prev_ctx->thread_info);
                gda_prev_ctx->thread_info = NULL;
            }
            DLT_UNREGISTER_CONTEXT (gda_prev_ctx->ctx->dlt_ctx);
            if (gda_prev_ctx == gp_gda_ctx)
            {
                gp_gda_ctx = gda_prev_ctx->next_ctx;
                if(gp_gda_ctx == NULL)
                {
                    /* Head node : unregister the GDA DLT default context and log functions */
                    gst_debug_remove_log_function(gda_get_debug_log);
#ifndef GST_VERSION_1
    gst_debug_add_log_function (gst_debug_log_default, NULL);
#else
	gst_debug_add_log_function (gst_debug_log_default, NULL, NULL);
#endif
                    DLT_UNREGISTER_CONTEXT (gda_dlt_ctx);
                }
                free(gda_prev_ctx);
                gda_prev_ctx = NULL;
                g_rec_mutex_unlock(&g_gda_mtx);
                return GDA_OK;
            }
            else
            {
                gda_cur_ctx->next_ctx = gda_prev_ctx->next_ctx;
                free(gda_prev_ctx);
                gda_prev_ctx = NULL;
                g_rec_mutex_unlock(&g_gda_mtx);
                return GDA_OK;
            }
        }
        else
        {
            gda_cur_ctx = gda_prev_ctx;
            gda_prev_ctx = gda_cur_ctx->next_ctx;
        }
    }
    g_rec_mutex_unlock(&g_gda_mtx);
    DLT_LOG (gda_dlt_ctx, DLT_LOG_ERROR, DLT_STRING("GDA : GDA Context not found for deletion"));
    return GDA_ER_CTX_NOT_REGISTERED;
}
